<?php
/**
 * Like Topic
 *
 * @package     AutomatorWP\Integrations\WP_Ulike\Triggers\Like_Topic
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_WP_Ulike_Like_Topic extends AutomatorWP_Integration_Trigger {

    public $integration = 'wp_ulike';
    public $trigger = 'wp_ulike_like_topic';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User likes or unlikes a topic', 'automatorwp-pro' ),
            'select_option'     => __( 'User likes or unlikes <strong>a topic</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Operation (likes or unlikes). %2$s: Post title. %3$s: Number of times. */
            'edit_label'        => sprintf( __( 'User %1$s %2$s %3$s time(s)', 'automatorwp-pro' ), '{operation}', '{post}', '{times}' ),
            /* translators: %1$s: Operation (likes or unlikes). %2$s: Post title. */
            'log_label'         => sprintf( __( 'User %1$s %2$s', 'automatorwp-pro' ), '{operation}', '{post}' ),
            'action'            => 'wp_ulike_after_process',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 4,
            'options'           => array(
                'operation' => array(
                    'from' => 'operation',
                    'fields' => array(
                        'operation' => array(
                            'name' => __( 'Operation:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'like'      => __( 'likes', 'automatorwp-pro' ),
                                'unlike'    => __( 'unlikes', 'automatorwp-pro' ),
                            ),
                            'default' => 'like'
                        ),
                    )
                ),
                'post' => automatorwp_utilities_post_option( array(
                    'name' => __( 'Topic:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'any topic', 'automatorwp-pro' ),
                    'post_type' => apply_filters( 'bbp_topic_post_type', 'topic' )
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_post_tags( __( 'Topic', 'automatorwp-pro' ) ),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int       $id         Liked element ID
     * @param string    $key        Like key
     * @param int       $user_id    User ID
     * @param string    $status     like|unlike
     */
    public function listener( $id, $key, $user_id, $status ) {

        // Bail if not is a topic like
        if( $key !== '_topicliked' ) {
            return;
        }

        // Trigger the like or unlike
        automatorwp_trigger_event( array(
            'trigger'   => $this->trigger,
            'user_id'   => $user_id,
            'post_id'   => $id,
            'status'    => $status,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Shorthand
        $operation = $trigger_options['operation'];

        if( empty( $operation ) ) {
            $operation = 'like';
        }

        // Don't deserve if post and status are not received
        if( ! isset( $event['post_id'] ) && ! isset( $event['status'] ) ) {
            return false;
        }

        // Don't deserve if post doesn't match with the trigger option
        if( ! automatorwp_posts_matches( $event['post_id'], $trigger_options['post'] ) ) {
            return false;
        }

        switch( $operation ) {
            case 'like':
                // Don't deserve if user hasn't liked
                if( $event['status'] !== 'like' ) {
                    return false;
                }
                break;
            case 'unlike':
                // Don't deserve if user hasn't unliked
                if( $event['status'] === 'like' ) {
                    return false;
                }
                break;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_WP_Ulike_Like_Topic();